// ascspg
//
// This program reads an ASCII file created by SPGASC, possible (likely!)
// edited, and writes a new SPG file.  That file may be loaded back into
// the scanner.
//
// A SPG file is the format written by GRE's Data Manager.  There are three
// major sections:
//
// o A five byte header, hex values 02 00 00 00 00
// o Actual data, 26496 bytes
// o A trailing filler of 6272 byes, all hex 00
//
// Written by Ken Plotkin
//            kjp15@cornell.edu
//            January 2001
//
// Translated from Fortran to C by Steve Falco
// 				   sfalco@worldnet.att.net
// 				   September 2001
// 				   This version is "Unix-centric" in that it
// 				   depends on various Unix system calls to
// 				   perform file I/O.  Debugged and tested on
// 				   RedHat Linux version 7.1.
//
// Distributed as careware.  If you like it, donate some money to a worthwhile
// charity.

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>

typedef unsigned char UCHAR;
typedef unsigned int UINT;

#define SMALL 32	// Small buffer size
#define HLEN 5		// Header length
#define BLEN 26496	// Body length
#define TLEN 6272	// Trailer length
#define EOL "\n"	// end of line

#define INFILE  "pro92in.asc"
#define OUTFILE "pro92out.spg"

int	find_type(char *btypin);
int	find_mode(char *cmodin);
int	find_freq(double freq);
void	use();
void	set_byte(int where, UCHAR what, char *why);
void	get_line(char *buf, int bsize, FILE *fp, char *where);
void	send_error(char *buf);
void	send_warning(char *buf);
void	chann(double freq, char *cmodin, char *pldpl, char *status,
		int *ifreq, int *ibytes);

int	hex1(int isize, int ia, int ib, int ic);

int	LineCount = 0;

UCHAR	*infile, *outfile;

double	freq0[] = { 29., 108., 137., 380., 806.  };
double	step[]  = { .0050, .0125, .0050, .0125, .0125 };
double	pl0[]   = { 0, 0, 51.2,76.8,102.4,128.0,153.6,179.2,204.8,230.4};
UCHAR	clndat[BLEN+SMALL];

UCHAR	cmode[][6] = {
	"AM   ","FM   ","PL   ","DPL  ","LTR  ","MOT  ", "EDACS"
};
UCHAR	btype[][6] = {
	"Conv ","     ","     ","     ","LTR  ","MOT  ", "EDACS"
};

main(int argc, char *argv[]) {

	extern char *optarg;
	extern int optind, opterr, optopt;

	int	i, i1, i2, ia, iarea, ib, ibnk, ibtype, ibyte3[SMALL];
	int	ibytes[SMALL], ic, ichan, icode, icont, idash;
	int	ifmap[SMALL], ifreq, ii, ii2, ii3, imode, intfrq, iprior;
	int	iradio, ireptr, iscdel, iscnm, isiz, itrdel;
	int	j, jchan, jj, k, kdum, kk, major, minor, nprepr, outfd;
	int	vmode;

	UINT	idval, itemp, ival2;

	FILE	*infp;

	UCHAR	actbank[SMALL], alpha[SMALL], bnkmod[SMALL], btypin[SMALL];
	UCHAR	chrbnk, cmodin[SMALL], *dashp, *dot, errmess[BUFSIZ];
	UCHAR	headtail[SMALL], in_line[BUFSIZ];
	UCHAR	pldpl[SMALL], status[SMALL], trnkid[SMALL], verbos[SMALL];

	double	code, delfrq, fracf, freq, freqhi, frequ, frqlck, scnmin;
	double	stepf, stepp, tone, vers;

	while((ic = getopt(argc, argv, "h")) != -1) {
		switch(ic) {
			case 'h': /*FALLTHROUGH*/
			case '?':
				use();
				break;
		}
	}

	if(optind < argc) {
		infile = argv[optind++];
	} else {
		infile = INFILE;
	}

	if(optind < argc) {
		outfile = argv[optind++];
	} else {
		outfile = OUTFILE;
	}

	// Keep the user out of trouble - part 1
	if(
		((dot = strrchr(infile, '.')) != 0) // we found the last dot
		&&				    // and
		( 				    // (
		    (strcmp(dot, ".spg") == 0)	    // we have .spg
		    ||				    // or
		    (strcmp(dot, ".SPG") == 0)	    // we have .spg
		)				    // )
	) {
		fprintf(stderr, "Oops - you're trying to convert a spg file (%s)." EOL, infile);
		fprintf(stderr, "This program converts an ASCII file to spg." EOL);
		fprintf(stderr, "Use spgasc to convert spg to ASCII." EOL);
		exit(1);
	}

	// Keep the user out of trouble - part 2
	// This won't catch every case, but it might help
	if(strcmp(infile, outfile) == 0) {
		fprintf(stderr, "Input file \"%s\" is the same as output file \"%s\"" EOL, infile, outfile);
		exit(1);
	}

	// Open the ASCII file, and check for version and whether it's verbose
	// mode.  So far, only verbose mode exists.
	if((infp = fopen(infile, "r")) == NULL) {
		fprintf(stderr, "Cannot open %s" EOL, infile);
		exit(1);
	}

	// Open the SPG output file
	if((outfd = open(outfile, O_WRONLY | O_CREAT, 0644)) == -1) {
		fprintf(stderr, "Cannot open %s" EOL, outfile);
		exit(1);
	}

	// Start the buffer off as empty - we should set every byte
	// below, but just in case...
	memset(clndat, 0, BLEN+SMALL);

	get_line(in_line, BUFSIZ, infp, "version line");
	if(3 != sscanf(in_line, "SPGASC Version %d.%d, %16s", &major, &minor, verbos)) {
		send_warning("cannot parse version line");
	}
	if(major != 1 || minor > 3) {
		send_error("wrong file version, this program is for ASC V 1.03 or earlier");
	}
	vmode = (verbos[0] == 'v') || (verbos[0] == 'V');
	vers = (double)major + ((double)minor * .01);

	// Top of "Look for next section" loop.  There are five types of such
	// sections: Banks, Weather Channels, Preprogrammed Search Bands,
	// Opening Message, and Scanner Settings
	while(1) {
		get_line(in_line, BUFSIZ, infp, "section header");
		if(strncmp(in_line, "******", 6)) {
			continue; // no match - read up to the next section
		}

		if(!strncmp(in_line, "****** Bank", 11)) {
			if(1 != sscanf(in_line, "****** Bank %d ****** Tag:", &k)) {
				send_warning("cannot parse bank start line");
			}
			get_line(in_line, BUFSIZ, infp, "blank line after section header");

			// Do the 50 channels for this bank
			for(j = 1; j <= 50; j++) {
				sprintf(errmess, "bank %d channel %d", k, j);
				get_line(in_line, BUFSIZ, infp, errmess);
				if(vmode) {
					if(6 != sscanf(in_line, "%d %*x %*x %*x %*x %*x %*x >%12c< %lf %5c %s %s", &jchan, alpha+1, &freq, cmodin, pldpl, status+1)) {
						send_warning("cannot parse bank channel");
					}
				} else {
					if(6 != sscanf(in_line, "%d >%12c< %lf %5c %s %s", &jchan, alpha+1, &freq, cmodin, pldpl, status+1)) {
						send_warning("cannot parse bank channel");
					}
				}
				cmodin[5] = 0;

				chann(freq, cmodin, pldpl, status,
					&ifreq, ibytes);

				// Put the results for this channel into CLNDAT
				jj = (k * 2443) + ((j-1) * 18);
				for(ii = 1; ii <= 6; ii++) {
					set_byte(jj+ii, (UCHAR)ibytes[ii], "bchan1");
				}
				for(ii = 1; ii <= 12; ii++) {
					set_byte(jj+6+ii, alpha[ii], "bchan2");
				}
			}

			// Bank search data
			get_line(in_line, BUFSIZ, infp, "blank line before bank search data");
			if(vmode) {
				get_line(in_line, BUFSIZ, infp, errmess);
				if(5 != sscanf(in_line, "Srch %*x %*x %*x %*x %*x %*x >%12c< %lf %5c %s %s", alpha+1, &freq, cmodin, pldpl, status+1)) {
					send_warning("cannot parse bank search");
				}
				get_line(in_line, BUFSIZ, infp, errmess);
				if(2 != sscanf(in_line, "End, step %*x %*x %*x %lf %lf", &freqhi, &stepp)) {
					send_warning("cannot parse bank search end");
				}
			} else {
				get_line(in_line, BUFSIZ, infp, errmess);
				if(5 != sscanf(in_line, "Srch >%12c< %lf %5c %s %s", alpha+1, &freq, cmodin, pldpl, status+1)) {
					send_warning("cannot parse bank search");
				}
				get_line(in_line, BUFSIZ, infp, errmess);
				if(2 != sscanf(in_line, "End, step %lf %lf", &freqhi, &stepp)) {
					send_warning("cannot parse bank search end");
				}
			}
			cmodin[5] = 0;

			chann(freq, cmodin, pldpl, status, &ifreq, ibytes);

			delfrq = freqhi - freq0[ifreq];
			intfrq = delfrq;
			fracf = delfrq - (double)intfrq;
			ibyte3[1] = intfrq;
			ibyte3[2] = (fracf + (.1 * step[ifreq])) / step[ifreq];
			ibyte3[3] = (stepp + (.1 * step[ifreq])) / step[ifreq];

			// Insert into CLNDAT
			jj = (k * 2443) + 900;
			for(ii = 1; ii <= 6; ii++) {
				set_byte(jj+ii, (UCHAR)(ibytes[ii]), "bsrch1");
			}
			for(ii = 1; ii <= 12; ii++) {
				set_byte(jj+6+ii, alpha[ii], "bsrch2");
			}
			for(ii = 1; ii <= 3; ii++) {
				set_byte(jj+18+ii, (UCHAR)(ibyte3[ii]), "bsrch3");
			}

			// Bank info.
			get_line(in_line, BUFSIZ, infp, "blank line before bank info");
			get_line(in_line, BUFSIZ, infp, errmess);
			if(strncmp(in_line+1, "Bank", 4)) {
				send_error("Bank info line not found");
			}

			// Bank alpha tag
			get_line(in_line, BUFSIZ, infp, errmess);
			if(vmode) {
				if(1 != sscanf(in_line+27, ">%12c<", alpha+1)) {
					send_warning("cannot parse bank alpha tag");
				}
			} else {
				if(1 != sscanf(in_line+7, ">%12c<", alpha+1)) {
					send_warning("cannot parse bank alpha tag");
				}
			}
			jj = (k * 2443) + 921;
			for(ii = 1; ii <= 12; ii++) {
				set_byte(jj+ii, alpha[ii], "batag");
			}

			// Bank mode and offset.  Flags in bits 0-1, 3-4 of
			// next byte
			get_line(in_line, BUFSIZ, infp, errmess);
			if(vmode) {
				if(1 != sscanf(in_line+28, "%23c", bnkmod+1)) {
					send_warning("cannot parse bank mode/offset");
				}
			} else {
				if(1 != sscanf(in_line+8, "%23c", bnkmod+1)) {
					send_warning("cannot parse bank mode/offset");
				}
			}
			ii = 0;
			if((bnkmod[1] == 'O') || (bnkmod[1] == 'o')) ii = ii+2;
			if( bnkmod[9] == '1') ii = ii+4;
			if( bnkmod[9] == '2') ii = ii+8;
			if( bnkmod[9] == '5') ii = ii+12;
			jj = (k * 2443) + 933;
			set_byte(jj+1, (UCHAR)ii, "bonoff");

			// Bank type.  Identify from list.
			get_line(in_line, BUFSIZ, infp, errmess);
			if(vmode) {
				if(1 != sscanf(in_line+28, "%5c", btypin)) {
					send_warning("cannot parse bank type");
				}
			} else {
				if(1 != sscanf(in_line+8, "%5c", btypin)) {
					send_warning("cannot parse bank type");
				}
			}
			btypin[5] = 0;
			ibtype = find_type(btypin);
			jj = (k * 2443) + 934;
			set_byte(jj+1, (UCHAR)ibtype, "btype");

			// Fleet map.  Straight read of data
			get_line(in_line, BUFSIZ, infp, errmess);
			if(vmode) {
				if(8 != sscanf(in_line+28, "S%d S%d S%d S%d S%d S%d S%d S%d",
					ifmap+1, ifmap+2, ifmap+3, ifmap+4, ifmap+5, ifmap+6, ifmap+7, ifmap+8)) {
					send_warning("cannot parse fleet map line");
				}
			} else {
				if(8 != sscanf(in_line+28, "S%d S%d S%d S%d S%d S%d S%d S%d",
					ifmap+1, ifmap+2, ifmap+3, ifmap+4, ifmap+5, ifmap+6, ifmap+7, ifmap+8)) {
					send_warning("cannot parse fleet map line");
				}
			}
			jj = (k * 2443) + 935;
			for(ii = 1; ii <= 8; ii++) {
				set_byte(jj+ii, (UCHAR)ifmap[ii], "fleet");
			}

			// Search lockouts.  Block of 100 bytes, two bytes per
			// frequency (integer offset, #steps from freq0).
			// FF FF if no lockout.  The ASC file has only those
			// frequencies that are locked out, so we begin by
			// setting the whole block to 0xff.
			jj = (k * 2443) + 943;
			for(ii = 1; ii <= 100; ii++) {
				set_byte(jj+ii, (UCHAR)0xff, "sloc0");
			}

			get_line(in_line, BUFSIZ, infp, "blank line before search lockout");
			get_line(in_line, BUFSIZ, infp, errmess);
			if(strncmp(in_line, " Search l", 8)) {
				send_error("Search lockouts not found");
			}
			while(1) {
				get_line(in_line, BUFSIZ, infp, errmess);
				if(!strncmp(in_line, " (end lockouts)", 15)) {
					break;
				}
				if(2 != sscanf(in_line, "%d %*x %*x %lf", &kk, &frqlck)) {
					send_warning("cannot parse search lockout line");
				}
				delfrq = frqlck - freq0[ifreq];
				intfrq = delfrq;
				fracf = delfrq - (double)intfrq;
				ibyte3[1] = intfrq;
				ibyte3[2] = (fracf + (0.1*step[ifreq])) / step[ifreq];
				jj = (k * 2443) + 943 + ((kk - 1) * 2);
				set_byte(jj+1, (UCHAR)ibyte3[1], "sloc1");
				set_byte(jj+2, (UCHAR)ibyte3[2], "sloc2");
			}

			// Talkgroup IDs.
			get_line(in_line, BUFSIZ, infp, "blank line before talkgroup IDs");
			get_line(in_line, BUFSIZ, infp, errmess);
			if(strncmp(in_line, " Talkgroup IDs:", 15)) {
				send_error("Talkgroup IDs not found");
			}
			for(kk = 1; kk <= 100; kk++) {
				get_line(in_line, BUFSIZ, infp, errmess);
				if(vers > 1.0) {
					if(4 != sscanf(in_line, "%d %x [%11c] >%12c<",
					  &kdum, &itemp, trnkid, alpha+1)) {
						send_warning("cannot parse talkgroup ID line");
					}
				} else {
					if(3 != sscanf(in_line, "%d %x >%12c<",
					  &kdum, &itemp, alpha+1)) {
						send_warning("cannot parse talkgroup ID line");
					}
				}
				ibytes[1] = itemp & 0xff;
				ibytes[2] = (itemp >> 8) & 0xff;

				// LTR ID.  Identify by it being LTR (btype 4)
				if((vers > 1.0) && (ibtype == 4)) {
					if(3 != sscanf(trnkid, "%1d%2d%3d", &iarea, &ireptr, &iradio)) {
						iarea = ireptr = iradio = 0;
					}
					ibytes[1] = iradio;
					ibytes[2] = ireptr + (32 * iarea);

					// 1-31-00 - Skip mode if text is "Skip"
					if(strstr(trnkid, "Skip")) {
						ibytes[2] = ibytes[2] + 128;
					}
				}

				// MOT Type 1 ID.  Detect by it being MOT and
				// there being a "-" in trnkid.
				if((dashp = strchr(trnkid, '-')) != 0) {
					idash = dashp - trnkid;
				} else {
					idash = -1;
				}
				if((vers > 1.0) && (ibtype == 5) && (idash > 0)) {
					if((idash) == 4) {
						if(3 != sscanf(trnkid, "%1d%3d-%1d", &ia, &ib, &ic)) {
							ia = ib = ic = 0;
						}
					}
					if((idash) == 3) {
						if(3 != sscanf(trnkid, "%1d%2d-%2d", &ia, &ib, &ic)) {
							ia = ib = ic = 0;
						}
					}
					isiz = ifmap[ia + 1];
					ival2 = hex1(isiz, ia, ib, ic);
					ibytes[1] = ival2 & 0xff;
					ibytes[2] = (ival2 >> 8) & 0xff;

					// Skip/Scan.  Because this was not in
					// the initial release, make "Scan"
					// the default.
					if(strstr(trnkid, "Skip")) {
						ibytes[2] = ibytes[2] | 0x80;
					}
				}

				// MOT Type 2 ID.  Detect by it being MOT and
				// not having a dash.  This will also detect
				// blank MOT IDs, which we need to check for
				// Skip status.  Blank MOT IDs can be type 1
				// or (as remnants) type 2, so "Skip" can be
				// in either the type 1 or 2 slots.
				if((vers > 1.0) && (ibtype == 5) && (idash <= 0)) {
					if(1 != sscanf(trnkid, "%d", &idval)) {
						idval = 0;
					}
					idval = idval >> 4;
					if(strstr(trnkid, "Skip")) {
						idval |= 0x8000;
					}
					ibytes[1] = idval & 0xff;
					ibytes[2] = (idval >> 8) & 0xff;
				}

				// EDACS ID
				if((vers > 1.0) && (ibtype == 6)) {
					if(1 != sscanf(trnkid, "%d", &idval)) {
						idval = 0;
					}
					if(strstr(trnkid, "Skip")) {
						idval |= 0x8000;
					}
					ibytes[1] = idval & 0xff;
					ibytes[2] = (idval >> 8) & 0xff;
				}

				jj = (k * 2443) + 1043 + ((kk - 1) * 14);
				set_byte(jj+1, (UCHAR)ibytes[1], "trnk1");
				set_byte(jj+2, (UCHAR)ibytes[2], "trnk2");
				for(ii = 1; ii <= 12; ii++) {
					set_byte(jj+2+ii, alpha[ii], "trnk3");
				}
			}
			continue;
		} // End of Bank processing

		if(!strncmp(in_line, "****** Weather", 14)) {
			get_line(in_line, BUFSIZ, infp, "blank line before weather bank");

			// Weather is Bank 10, with 10 channels.
			k = 10;
			for(j = 1; j <= 10; j++) {
				sprintf(errmess, "bank %d channel %d", k, j);
				get_line(in_line, BUFSIZ, infp, errmess);
				if(vmode) {
					if(6 != sscanf(in_line, "%d %*x %*x %*x %*x %*x %*x >%12c< %lf %5c %s %s",
						   &jchan, alpha+1, &freq, cmodin, pldpl, status+1)) {
						send_warning("cannot parse weather channel");
					}
				} else {
					if(6 != sscanf(in_line, "%d >%12c< %lf %5c %s %s",
						   &jchan, alpha+1, &freq, cmodin, pldpl, status+1)) {
						send_warning("cannot parse weather channel");
					}
				}
				cmodin[5] = 0;

				chann(freq, cmodin, pldpl, status, &ifreq, ibytes);

				// Put the results for this channel into CLNDAT
				jj = (k * 2443) + ((j - 1) * 18);
				for(ii = 1; ii <= 6; ii++) {
					set_byte(jj+ii, (UCHAR)ibytes[ii], "wchan1");
				}
				for(ii = 1; ii <= 12; ii++) {
					set_byte(jj+6+ii, alpha[ii], "wchan2");
				}
			}
			continue;
		} // End of weather processing

		if(!strncmp(in_line, "****** Preprogrammed", 20)) {
			get_line(in_line, BUFSIZ, infp, "blank line before Preprogrammed Search Bands");

			// This is Bank 10, with number of channels to be determined
			k = 10;
			j = 0;
			while(1) {
				j++;
				sprintf(errmess, "bank %d channel %d", k, j);
				get_line(in_line, BUFSIZ, infp, errmess);
				if(strlen(in_line) < 8) {
					break;	// This section ends with a
						// blank line - any valid
						// line has about 70
						// characters
				}
				if(vmode) {
					if(6 != sscanf(in_line, "%d %*x %*x %*x %*x %*x %*x >%12c< %lf %5c %lf %lf",
						   &jchan, alpha+1, &freq, cmodin, &frequ, &stepf)) {
						send_warning("cannot parse search band");
					}
				} else {
					if(6 != sscanf(in_line, "%d >%12c< %lf %5c %lf %lf",
						   &jchan, alpha+1, &freq, cmodin, &frequ, &stepf)) {
						send_warning("cannot parse search band");
					}
				}
				cmodin[5] = 0;

				// Decode the mode
				imode = find_mode(cmodin);

				// Frequency range
				ifreq = find_freq(freq);

				// Create Byte 4: low nybble is ifreq, high is
				// imode (Like regular channel Byte 3, with
				// nybbles reversed)
				ibytes[4] = (imode * 16) + ifreq;

				// Get whole and fractional part of frequency,
				// relative to freq0
				delfrq = freq - freq0[ifreq];
				intfrq = delfrq;
				fracf = delfrq - (double)intfrq;

				// Byte 1 is the whole number offset, Byte 2
				// is the number of steps (Just like regular
				// channel)
				ibytes[1] = intfrq;
				ibytes[2] = (fracf + (0.1 * step[ifreq])) / step[ifreq];

				// Bytes 5 and 6 similar, for the upper frequency
				delfrq = frequ - freq0[ifreq];
				intfrq = delfrq;
				fracf = delfrq - (double)intfrq;
				ibytes[5] = intfrq;
				ibytes[6] = (fracf + (0.1 * step[ifreq])) / step[ifreq];

				// Byte 3 is number of frequency steps in search step
				ibytes[3] = (stepf + (0.1 * step[ifreq])) / step[ifreq];

				// If the mode or freq are bogus, mark the
				// channel as unused.  SAF 2001-10-01
				if(imode == -1 || ifreq == -1) {
					memset(ibytes, 0, 7);
					memset(alpha, 0, 13);
					ibytes[3] = 0x01;
					ibytes[4] = 0x10;
				}

				// Put into CLNDAT.
				jj = (k * 2443) + 180 + ((j - 1) * 18);
				for(ii = 1; ii <= 6; ii++) {
					set_byte(jj+ii, (UCHAR)ibytes[ii], "psrch1");
				}
				for(ii = 1; ii <= 12; ii++) {
					set_byte(jj+6+ii, alpha[ii], "psrch2");
				}
			}

			// Total number.  Insert into proper slot in CLNDAT.
			nprepr = j - 1;
			set_byte(0x6769+1, (UCHAR)nprepr, "psrch3");

			// If we don't have 100, fill out the rest with null data
			if(nprepr < 100) {
				for(j =  nprepr + 1; j <= 100; j++) {
					jj = (k * 2443) + 180 + ((j - 1) * 18);
					set_byte(jj+1, 0, "psrch4");
					set_byte(jj+2, 0, "psrch5");
					set_byte(jj+3, 1, "psrch6");
					set_byte(jj+4, 16, "psrch7");
					for(ii = 5; ii <= 18; ii++) {
						set_byte(jj+ii, 0, "psrch8");
					}
				}
			}
			continue;
		} // End of preprogrammed search banks

		// Opening screen message
		if(!strncmp(in_line, "****** Opening", 14)) {
			get_line(in_line, BUFSIZ, infp, "blank line before Opening screen message");

			for(i = 1; i <= 4; i++) {
				sprintf(errmess, "opening message line %d", i);
				get_line(in_line, BUFSIZ, infp, errmess);
				if(1 != sscanf(in_line, " >%12c<", alpha+1)) {
					send_warning("cannot parse opening message line");
				}

				jj = 26410 + ((i - 1) * 12);
				for(j = 1; j <= 12; j++) {
					set_byte(jj+j, alpha[j], "oscr");
				}
			}
			continue;
		} // End of Opening screen message

		// Scanner settings
		if(!strncmp(in_line, "****** Scanner", 14)) {
			get_line(in_line, BUFSIZ, infp, "blank line before Scanner settings");

			jj = 0x675a;
			sprintf(errmess, "scanner setting");

			// Unknown, default 14.  Read the hex byte and pass it on.
			get_line(in_line, BUFSIZ, infp, errmess);
			if(1 != sscanf(in_line+12, "%x", &ii)) {
				send_warning("cannot parse unknown byte 1");
			}
			set_byte(jj+1, (UCHAR)ii, "u1");
	   
			// Backlight on time, seconds
			get_line(in_line, BUFSIZ, infp, errmess);
			if(1 != sscanf(in_line+12, "%d", &ii)) {
				send_warning("cannot parse Backlight on time");
			}
			set_byte(jj+2, (UCHAR)ii, "blt");
	   
			// Scan delay time, msec.  Resolution of 100 msec
			get_line(in_line, BUFSIZ, infp, errmess);
			if(1 != sscanf(in_line+12, "%d", &iscdel)) {
				send_warning("cannot parse Scan delay time");
			}
			set_byte(jj+3, (UCHAR)(iscdel / 100), "sdt");

			// Trunk delay time, msec.  Resolution of 100 msec
			get_line(in_line, BUFSIZ, infp, errmess);
			if(1 != sscanf(in_line+12, "%d", &itrdel)) {
				send_warning("cannot parse Trunk delay time");
			}
			set_byte(jj+4, (UCHAR)(itrdel / 100), "tdt");

			// Unknown default 3F.  Read the hex byte and pass it on.
			get_line(in_line, BUFSIZ, infp, errmess);
			if(1 != sscanf(in_line+12, "%x", &ii)) {
				send_warning("cannot parse unknown byte 5");
			}
			set_byte(jj+5, (UCHAR)ii, "u5");

			// Minimum scan delay time, resolution of .1 sec
			get_line(in_line, BUFSIZ, infp, errmess);
			if(1 != sscanf(in_line+12, "%lf", &scnmin)) {
				send_warning("cannot parse Minimum scan delaytime");
			}
			iscnm = (10.0 * scnmin) + 0.5;
			set_byte(jj+6, (UCHAR)iscnm, "msdt");

			// Display contrast, 9-14
			get_line(in_line, BUFSIZ, infp, errmess);
			if(1 != sscanf(in_line+12, "%d", &icont)) {
				send_warning("cannot parse Display contrast");
			}
			set_byte(jj+7, (UCHAR)icont, "dc");

			// Priority channel (bank, channel)
			get_line(in_line, BUFSIZ, infp, errmess);
			if(2 != sscanf(in_line+12, "%d %d", &ibnk, &ichan)) {
				send_warning("cannot parse Priority channel");
			}
			set_byte(jj+8, (UCHAR)ichan, "pc1");
			set_byte(jj+9, (UCHAR)ibnk, "pc2");

			// Last search frequency
			get_line(in_line, BUFSIZ, infp, errmess);
			if(3 != sscanf(in_line+12, "%x %x %x", &ii, &ii2, &ii3)) {
				send_warning("cannot parse Last search frequency");
			}
			set_byte(jj+10, (UCHAR)ii, "lsf1");
			set_byte(jj+11, (UCHAR)ii2, "lsf2");
			set_byte(jj+12, (UCHAR)ii3, "lsf3");

			// Priority on/off
			get_line(in_line, BUFSIZ, infp, errmess);
			if(1 != sscanf(in_line+12, "%x", &iprior)) {
				send_warning("cannot parse Priority on/off");
			}
			set_byte(jj+13, (UCHAR)iprior, "poo");

			// WX Priority on/off
			get_line(in_line, BUFSIZ, infp, errmess);
			if(1 != sscanf(in_line+12, "%x", &iprior)) {
				send_warning("cannot parse WX Priority on/off");
			}
			set_byte(jj+14, (UCHAR)iprior, "wxpoo");

			// WX Priority channel number
			get_line(in_line, BUFSIZ, infp, errmess);
			if(1 != sscanf(in_line+12, "%x", &iprior)) {
				send_warning("cannot parse WX Priority channel number");
			}
			set_byte(jj+15, (UCHAR)iprior, "wxpcn");

			// Number of preprogrammed search bands; already computed actual.  Skip.
			get_line(in_line, BUFSIZ, infp, "Number of preprogrammed search bands");

			// Keypad tone frequency
			get_line(in_line, BUFSIZ, infp, errmess);
			if(1 != sscanf(in_line+12, "%lf", &tone)) {
				send_warning("cannot parse Keypad tone frequency");
			}
			code = 1000000.0 / tone;
			icode = -(int)code;
			i1 = icode & 0xff;
			i2 = (icode >> 8) & 0xff;
			set_byte(jj+17, (UCHAR)i1, "ktf1");
			set_byte(jj+18, (UCHAR)i2, "ktf2");

			// Enabled scan banks.  Copy 0-9 over, change anything else to A5
			get_line(in_line, BUFSIZ, infp, errmess);
			if(1 != sscanf(in_line+12, "%s", actbank+1)) {
				send_warning("cannot parse Enabled scan banks");
			}
			for(ii = 1; ii <= 10; ii++) {
				chrbnk = actbank[ii];
				if((chrbnk >= '0') && (chrbnk <= '9')) {
					set_byte(jj+18+ii, chrbnk, "escb1");
				} else {
					set_byte(jj+18+ii, 0xA5, "escb2");
				}
			}

			// Enabled search banks.  Same deal.
			get_line(in_line, BUFSIZ, infp, errmess);
			if(1 != sscanf(in_line+12, "%s", actbank+1)) {
				send_warning("cannot parse Enabled search banks");
			}
			for(ii = 1; ii <= 10; ii++) {
				chrbnk = actbank[ii];
				if((chrbnk >= '0') && (chrbnk <= '9')) {
					set_byte(jj+28+ii, chrbnk, "eseb1");
				} else {
					set_byte(jj+28+ii, 0xA5, "eseb2");
				}
			}
		} // End of Scanner Settings
		break;
	}

	// That's it.  Write the header
	headtail[0] = 2;
	headtail[1] = 0;
	headtail[2] = 0;
	headtail[3] = 0;
	headtail[4] = 0;
	if(5 != write(outfd, headtail, 5)) {
		fprintf(stderr, "Cannot write header to %s" EOL, outfile);
		exit(1);
	}

	// Write the data
	if(BLEN != write(outfd, clndat+1, BLEN)) {
		fprintf(stderr, "Cannot write data to %s" EOL, outfile);
		exit(1);
	}
	
	// Write the trailer
	memset(clndat, 0, TLEN);
	if(TLEN != write(outfd, clndat, TLEN)) {
		fprintf(stderr, "Cannot write trailer to %s" EOL, outfile);
		exit(1);
	}

	close(outfd);
	exit(0);
}

int
find_type(char *btypin)
{
	int ii, slen, ibtype = -1;

	slen = strlen(btypin);
	for(ii = 0; ii <= 6; ii++) {
		if(!strncmp(btypin, btype[ii], slen)) {
			ibtype = ii;
		}
	}
	if(ibtype == -1) {
		send_warning("cannot match bank type");
	}

	return ibtype;
}

int
find_mode(char *cmodin)
{
	int ii, slen, imode = -1;

	slen = strlen(cmodin);
	for(ii = 0; ii <= 6; ii++) {
		if(!strncmp(cmodin, cmode[ii], slen)) {
			imode = ii;
		}
	}
	if(imode == -1) {
		send_warning("cannot match transmission mode");
	}

	return imode;
}

int
find_freq(double freq)
{
	int ii, ifreq = -1;

	// Frequency range - don't jump out early, we want the highest band
	// that is less than the desired frequency
	for(ii = 0; ii <= 4; ii++) {
		if(freq >= freq0[ii]) {
			ifreq = ii;
		}
	}
	if(ifreq == -1) {
		send_warning("cannot match frequency band");
	}

	return ifreq;
}

// chann() encodes input frequency, mode, pl/dpl and status into the six
// bytes which will go into the SPG file.  It returns these as integer values
// in IBYTES.  The frequency range, IFREQ, is also returned.
void
chann(double freq, char *cmodin, char *pldpl, char *status, int *ifreq, int *ibytes)
{
	UINT idpl;
	int imode, i, intfrq, ipl;
	double delfrq, fracf, plfreq, plfrac;

	// Transmission mode.  Do this first since we'll need it if this is an
	// unused channel
	imode = find_mode(cmodin);

	// Byte 4 is flags.  Do these here because if it's an unused channel
	// we set the data bytes to 00 00 (imode) (byte4) 00 00 and return
	// immediately.
	ibytes[1] = 0;
	ibytes[2] = 0;
	ibytes[3] = imode;
	ibytes[4] = 0;
	ibytes[5] = 0;
	ibytes[6] = 0;

	*ifreq = 0;

	if(status[1] == 'U') ibytes[4] |= 8;
	if(status[2] == 'D') ibytes[4] |= 4;
	if(status[3] == 'A') ibytes[4] |= 2;
	if(status[4] == 'L') ibytes[4] |= 1;

	if(status[1] == 'U') return;

	*ifreq = find_freq(freq);

	// Create Byte 3: high nybble is ifreq, low is imode
	ibytes[3] = (*ifreq << 4) | imode;

	// Get whole and fractional part of frequency, relative to freq0
	delfrq = freq - freq0[*ifreq];
	intfrq = delfrq;
	fracf = delfrq - (double)intfrq;

	// Byte 1 is the whole number offset, Byte 2 is the number of steps
	// Note the diddle of the fractional part, to ensure we don't lose
	// a step due to rounding.
	ibytes[1] = intfrq;
	ibytes[2] = (fracf + (0.5 * step[*ifreq])) / step[*ifreq];

	// Bytes 5 and 6 are PL or DPL tones.
	//   PL is a base and step offset thing.
	//   DPL is a 3-digit octal value.
	//   Both bytes are 0 if it's neither PL nor DPL
	ibytes[5] = 0;
	ibytes[6] = 0;

	if(imode == 2) { // PL
		sscanf(pldpl, "%lf", &plfreq);
		ipl = 0;
		for(i = 2; i <= 9; i++) {
			if(plfreq >= pl0[i]) {
				ipl = i;
			}
		}
		if(ipl != 0) {
			plfrac = plfreq - pl0[ipl];
			ibytes[6] = ipl;
			ibytes[5] = (int)((plfrac * 10.0) + 0.5);
		}
        }

	if(imode == 3) { // DPL
		sscanf(pldpl, "D%o", &idpl);
		ibytes[6] = idpl >> 8;
		ibytes[5] = idpl & 0xff;
        }

	return;
}

// This routine computes the hex value corresponding to Motorola Type 1
// block (ia), fleet (ib) and subfleet (ic), for size Snn, nn = isize.
int
hex1(int isize, int ia, int ib, int ic)
{
	static int nsubs[] = {  0,  4,  8,  8, 16,  4,  8,  4,
				4,  4,  8, 16, 16, 16, 16,  0 };
	static int incrs[] = {  0,  1,  4,  8, 32,  2,  2,  4,
				8, 16, 16, 16, 64,128,256,  0 };
	int nsub, incr, ival;

	nsub = nsubs[isize & 0x0f];
	incr = incrs[isize & 0x0f];

	ival = (ia * 0x200) + (ib * incr * nsub) + (ic * incr) + (incr - 1);
	return ival;
}

void
set_byte(int where, UCHAR what, char *why)
{
#ifdef DEBUG
	fprintf(stderr, "0x%04x: 0x%02x (%s)\n", where+4, what, why);
#endif
	clndat[where] = what;
}

void
get_line(char *buf, int bsize, FILE *fp, char *where)
{
	char errormess[BUFSIZ];
	char *bp = buf;

	LineCount++;
	for(/**/; (*bp = getc(fp)) != EOF; bp++) {
		if(*bp == '\n') { // Unix just has \n, dos has \r\n
			*++bp = 0;
			return;
		}
		if(*bp == 0) {
			*bp = ' ';
		}
	}
	sprintf(errormess, "End of file reading %s" EOL, where);
	send_error(errormess);
	/*NOTREACHED*/

}

void
send_error(char *buf)
{
	fprintf(stderr, "Error: %s at line %d, file %s" EOL,
			buf, LineCount, infile);
	exit(1);
	/*NOTREACHED*/
}

void
send_warning(char *buf)
{
	fprintf(stderr, "Warning: %s at line %d, file %s" EOL,
			buf, LineCount, infile);
}

void
use()
{
	fprintf(stderr, "ascspg [-h] [inputfile] [outputfile.spg]" EOL);
	fprintf(stderr, "  -h prints this help message" EOL);
	fprintf(stderr, "  inputfile is the ascii source file" EOL);
	fprintf(stderr, "          (defaults to %s)" EOL, INFILE);
	fprintf(stderr, "  outputfile.spg is the binary output file" EOL);
	fprintf(stderr, "          (defaults to %s, name must end with .spg)" EOL, OUTFILE);
}
